/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <dirent.h>
#ifdef HAVE_MMAN_H
#include <sys/mman.h>
#endif

#if (WIN32|WINNT)
#include <process.h>
#endif

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_spell.h"
#include "udm_cache.h"
#include "udm_crc32.h"
#include "udm_boolean.h"
#include "udm_searchtool.h"
#include "udm_agent.h"
#include "udm_xmalloc.h"
#include "udm_stopwords.h"
#include "udm_proto.h"
#include "udm_vars.h"
#include "udm_mutex.h"
#include "udm_conf.h"
#include "udm_doc.h"
#include "udm_db_int.h"
#include "udm_vars.h"
#include "udm_log.h"

#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

#ifdef HAVE_MERGESORT
#define UdmSort mergesort
#else
#define UdmSort qsort
#endif

#define DEBUG_SEARCH 1

/* uncomment this to enable MODE_ALL realisation via search limits */
/*#define MODE_ALL_VIA_LIMITS*/

static int open_host(char *hostname,int port, int timeout)
{
	int net;
	struct hostent *host;
	struct sockaddr_in sa_in;

	bzero((char*)&sa_in,sizeof(sa_in));

	if (port){
		sa_in.sin_port= htons((u_short)port);
	}else{
		return(UDM_NET_ERROR);
	}

	if ((sa_in.sin_addr.s_addr=inet_addr(hostname)) != INADDR_NONE){
		sa_in.sin_family=AF_INET;
	}else{
		host=gethostbyname(hostname);
		if (host){
			sa_in.sin_family=host->h_addrtype;
			memcpy(&sa_in.sin_addr, host->h_addr, (size_t)host->h_length);
		}else{
			return(UDM_NET_CANT_RESOLVE);
		}
	}
	net=socket(AF_INET, SOCK_STREAM, 0);

	if(connect(net, (struct sockaddr *)&sa_in, sizeof (sa_in)))
		return(UDM_NET_CANT_CONNECT);

	return(net);
}

typedef struct{
	uint4 url_id;
	uint4 wrd_id;
	uint4 coord;
} T_URL_WRD_CRD;

typedef struct{
	uint4	wrd_id;
	off_t	pos;
	size_t	len;
} T_DAT_IND;

static int cmp_url_id(const UDM_URL_CRD *t1, const UDM_URL_CRD *t2){
	if(t1->url_id<t2->url_id) return(-1);
	if(t1->url_id>t2->url_id) return(1);
	{
		uint4 n1=UDM_WRDPOS(t1->coord);
		uint4 n2=UDM_WRDPOS(t2->coord);
		if(n1<n2)return(-1);
		if(n1>n2)return(1);
	}
	return(0);
}

static int cmp_ind(const T_DAT_IND *t1, const T_DAT_IND *t2){
	if(t1->wrd_id<t2->wrd_id) return(-1);
	if(t1->wrd_id>t2->wrd_id) return(1);
	return(0);
}

static int cmp_uint4(const uint4 *u1, const uint4 *u2){
	if(*u1<*u2) return(-1);
	if(*u1>*u2) return(1);
	return(0);
}
         
/***********************************/

#define LOGDIR	"raw"
#define TREEDIR	"tree"
#define SPLDIR  "splitter"

/******** Convert category string into 32 bit number *************/
void UdmDecodeHex8Str(const char *hex_str,uint4 *hi,uint4 *lo){
  char str[33],str_hi[17],str_lo[17];

        strncpy(str, hex_str, 16);
	str[16] = '\0';
	strcat(str,"0000000000000000");
	strncpy(str_hi,&str[0],8); str_hi[8]=0;
	strncpy(str_lo,&str[8],8); str_lo[8]=0;
	
	*hi=strtoul(str_hi, (char **)NULL, 16);
	*lo=strtoul(str_lo, (char **)NULL, 16);
}

/*************************** Sort functions **************************/
/* Function to sort LOGWORD list in (wrd_id,url_id,coord,time_stamp) order */
static int cmplog(const UDM_LOGWORD *s1,const UDM_LOGWORD *s2){
	if(s1->wrd_id<s2->wrd_id)return(-1);
	if(s1->wrd_id>s2->wrd_id)return(1);

	if(s1->url_id<s2->url_id)return(-1);
	if(s1->url_id>s2->url_id)return(1);

	if(s1->coord<s2->coord)return(-1);
	if(s1->coord>s2->coord)return(1);

	if(s2->stamp<s1->stamp)return(-1);
	if(s2->stamp>s1->stamp)return(1);

	return(0);
}

int UdmOpenCache(UDM_AGENT *A,void *vdb){
	char		*host,*prt;
	int		port=UDM_LOGD_PORT;
	UDM_DB		*db=vdb;
	const char	*logd_addr=UdmVarListFindStr(&A->Conf->Vars,"LogdAddr",NULL);
	
	if(db->DBMode!=UDM_DBMODE_CACHE)return UDM_OK;
	
	if(logd_addr){
		host=strdup(logd_addr);
		if((prt=strchr(host,':'))){
			*prt='\0';
			if(!(port=atoi(prt+1)))
				port=UDM_LOGD_PORT;
		}
		if((db->logd_fd=open_host(host,port,60)) < 0){
			sprintf(db->errstr,"Can't connect to cachelogd at %s:%d",host,port);
			db->errcode=1;
			UDM_FREE(host);
			db->logd_fd=0;
			strcpy(A->Conf->errstr,db->errstr);
			A->Conf->errcode=db->errcode;
			return UDM_ERROR;
		}
		UDM_FREE(host);
	}else{
		if(UDM_OK!=LogdInit(db,A->Conf->vardir)){
			strcpy(A->Conf->errstr,db->errstr);
			A->Conf->errcode=db->errcode;
			return UDM_ERROR;
		}
	}
	return(UDM_OK);
}

int UdmCloseCache(UDM_DB *db){
	if(db->logd_fd>0)closesocket(db->logd_fd);
	return LogdClose(db);
}

static int PresentInDelLog(UDM_LOGDEL *buf, int buf_count, int *start, const int url_id){
	register int m,l,r;
	
	if(start)l=*start;
	else l=0;
	r=buf_count;
	while(l<r){
		m=(l+r)/2;
		if(buf[m].url_id<url_id) l=m+1;
		else r=m;
	}
	if(start)*start=r;
	if(r==buf_count) return(0);
	else
		if(buf[r].url_id==url_id) return(buf[r].stamp);
		else return(0);
}

/***** Write a marker that the old content of url_id should be deleted ****/
int UdmDeleteURLFromCache(UDM_AGENT * Indexer,int url_id,UDM_DB *db){
	int sent;
	UDM_LOGD_CMD cmd;
	char reply;
	cmd.stamp=time(NULL);
	cmd.url_id=url_id;
	cmd.cmd=0;
	cmd.nwords=0;

	if(db->logd_fd){
		sent = UdmSend(db->logd_fd, &cmd, sizeof(cmd), 0);
		if(sent!=sizeof(cmd)){
			sprintf(db->errstr,"Can't write to cachelogd: %s",strerror(errno));
			return(UDM_ERROR);
		}
		if (UdmRecvall(db->logd_fd, &reply, sizeof(char)) != 1) {
		  return(UDM_ERROR);
		}
		if (reply != 'O') {
		  return UDM_ERROR;
		}
	}else{
		if(LogdStoreDoc(db,cmd,NULL)){
			return(UDM_ERROR);
		}
	}
	return(UDM_OK);
}

int UdmStoreWordsCache(UDM_AGENT * Indexer,UDM_DOCUMENT *Doc,UDM_DB *db){
	int		sent;
	size_t		i;
	UDM_LOGD_CMD	cmd;
	UDM_LOGD_WRD	*wrd;
	char		reply;
	int		url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	
	/* Mark that old content should be deleted    */
	/* Note that this should be done every time   */
	/* even if previous status=0, i.e. first time */
	/* indexing                                   */
	
	if ( (cmd.nwords = Doc->Words.nwords) == 0) {
	  return (UDM_OK);
	}

	cmd.stamp=time(NULL);
	cmd.url_id=url_id;
	cmd.cmd=0;
	
	wrd=(UDM_LOGD_WRD*)malloc(cmd.nwords*sizeof(UDM_LOGD_WRD));
	for(i=0;i<Doc->Words.nwords;i++){
		wrd[i].coord=Doc->Words.Word[i].coord;
		wrd[i].wrd_id=UdmStrCRC32(Doc->Words.Word[i].word);
	}
	if(db->logd_fd){
		sent = UdmSend(db->logd_fd, &cmd, sizeof(cmd), 0);
		if(sent!=sizeof(cmd)){
			sprintf(db->errstr,"Can't write to cachelogd: %s",strerror(errno));
			return(UDM_ERROR);
		}
		sent = UdmSend(db->logd_fd, wrd, cmd.nwords * sizeof(UDM_LOGD_WRD), 0);
		if(sent!=cmd.nwords*sizeof(UDM_LOGD_WRD)){
			sprintf(db->errstr,"Can't write to cachelogd: %s",strerror(errno));
			return(UDM_ERROR);
		}
		if (UdmRecvall(db->logd_fd, &reply, sizeof(char)) != 1) {
		  return(UDM_ERROR);
		}
		if (reply != 'O') {
		  return UDM_ERROR;
		}
	}else{
		if(LogdStoreDoc(db,cmd,wrd)){
			return(UDM_ERROR);
		}
	}
	free(wrd);
	return(UDM_OK);
}

/***************** split one log into cache *****************/

/* 
This function removes duplicate
(wrd_id,url_id) pairs using timestamps.
Most new pairs will remane.
*/

__INDLIB__ int UdmRemoveDelLogDups(UDM_LOGDEL *words, int n){
	int i,j;
	
	j=0;
	for(i=1;i<n;i++){
		if(words[j].url_id!=words[i].url_id){
			j++;
			if(i!=j)words[j]=words[i];
		}
	}
	return(j+1);
}

/* 
This function removes non-fresh  
words from UDM_LOGWORD array using timestamps
from delete log. Only those (word_id,url_id)
pairs which are newer than (url_id) from 
delete log will remane in "words" array 
*/

static size_t RemoveOldWords(UDM_LOGWORD *words, size_t n,UDM_LOGDEL *del, int del_count){
	size_t i,j=0;
	
	for(i=1;i<n;i++){
		if((words[j].wrd_id!=words[i].wrd_id)
		   ||(words[j].url_id!=words[i].url_id)
		   ||(words[j].coord!=words[i].coord)
		){
			if(PresentInDelLog(del,del_count,NULL,words[i].url_id)<=words[i].stamp){
				j++;
				if(i!=j){
					words[j]=words[i];
				}
			}
		}
	}
	/* Check first word */
	if(j==0){
		if(PresentInDelLog(del,del_count,NULL,words[0].url_id)>words[0].stamp)
			return(0);
	}
	return(j+1);
}

int UdmClearCacheTree(UDM_ENV * Conf){
	int i;
	char fname[1024];
	
	for(i=0;i<0x1000;i++){
		snprintf(fname,sizeof(fname),"%s%s%c%03X.dat",Conf->vardir,TREEDIR,UDMSLASH,i);
		unlink(fname);
		snprintf(fname,sizeof(fname),"%s%s%c%03X.ind",Conf->vardir,TREEDIR,UDMSLASH,i);
		unlink(fname);
	}
	return(0);
}

typedef struct{
	uint4 wrd_id;
	uint4 pos;
	uint4 len;
	int   done;
} T_WRD_POS_LEN;

static int PresentInNew(T_WRD_POS_LEN *keys, size_t nkeys, uint4 wrd_id){
	size_t l,r,m;
	
	l=0;
	r=nkeys;
	while(l<r){
		m=(l+r)/2;
		if(keys[m].wrd_id<wrd_id) l=m+1;
		else r=m;
	}
	if(r==nkeys) return(-1);
	else
		if(keys[r].wrd_id==wrd_id) return(r);
		else return(-1);
}

__INDLIB__ int UdmSplitCacheLog(UDM_ENV * Env,int log,UDM_LOGDEL *del_buf,int del_count){
	size_t bytes;
	char log_name[1024];
	char old_dat_name[1024];
	char new_dat_name[1024];
	char old_ind_name[1024];
	char new_ind_name[1024];
	struct stat sb;
	UDM_LOGWORD *log_buf = NULL;

	int log_fd=0;
	int old_dat_fd=0, new_dat_fd=0;
	int old_ind_fd=0, new_ind_fd=0;
	
	size_t	old_ind_count=0;
	T_DAT_IND *old_ind=NULL;
	
	size_t new_ind_count=0;
	size_t new_ind_mcount=500;
	T_DAT_IND *new_ind=(T_DAT_IND*)malloc(new_ind_mcount*sizeof(T_DAT_IND));

printf("Log %03X ",log); fflush(stdout);

	snprintf(old_dat_name, sizeof(old_dat_name), "%s%s%c%03X%s", Env->vardir, TREEDIR, UDMSLASH, log, ".dat");
	old_dat_fd=open(old_dat_name,O_RDONLY|UDM_BINARY);
	snprintf(old_ind_name, sizeof(old_dat_name), "%s%s%c%03X%s", Env->vardir, TREEDIR, UDMSLASH, log, ".ind");
	old_ind_fd=open(old_ind_name,O_RDONLY|UDM_BINARY);
	
	if((old_dat_fd>0)&&(old_ind_fd>0)){
		fstat(old_ind_fd, &sb);
		//ind=(T_DAT_IND*)mmap(0,sb.st_size,PROT_READ,MAP_SHARED,ind_fd,0);
		old_ind=(T_DAT_IND*)malloc((size_t)sb.st_size);
		read(old_ind_fd,old_ind,(size_t)sb.st_size);
		close(old_ind_fd);
		old_ind_count=sb.st_size/sizeof(T_DAT_IND);
	}

	/* Open log file */
	snprintf(log_name, sizeof(log_name), "%s%s%c%03X.log", Env->vardir, SPLDIR, UDMSLASH, log);
	if((log_fd=open(log_name,O_RDONLY|UDM_BINARY))<0){
	  if (errno == ENOENT) {
printf("No new... "); fflush(stdout);
	  } else {
		UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't open '%s': (%d) %s", log_name, errno, strerror(errno));
/*		goto err1;*/
	  }
	}
	snprintf(new_dat_name, sizeof(new_dat_name), "%s%s%c%03X%s", Env->vardir, TREEDIR,UDMSLASH, log, ".dat.new");
	if((new_dat_fd=open(new_dat_name,O_CREAT|O_WRONLY|O_TRUNC|UDM_BINARY,UDM_IWRITE))<0){
		UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't open '%s': %s", new_dat_name, strerror(errno));
		goto err1;
	}
	snprintf(new_ind_name, sizeof(new_dat_name), "%s%s%c%03X%s", Env->vardir, TREEDIR, UDMSLASH, log, ".ind.new");
	if((new_ind_fd=open(new_ind_name,O_CREAT|O_WRONLY|O_TRUNC|UDM_BINARY,UDM_IWRITE))<0){
		UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't open '%s': %s\n",new_ind_name,strerror(errno));
		goto err1;
	}

	{
		size_t mkeys=500;
		size_t nkeys=0;
		T_WRD_POS_LEN *keys=(T_WRD_POS_LEN*)malloc(mkeys*sizeof(T_WRD_POS_LEN));
		size_t i,prev=0;
		size_t n;
		size_t k;
		size_t pos=0;
		

		/* Allocate buffer for log */
		if (log_fd > 0) { 
printf("reading... "); fflush(stdout);
		  fstat(log_fd, &sb);
		  log_buf=(UDM_LOGWORD*)malloc((size_t)sb.st_size);
		  bytes=read(log_fd,log_buf,(size_t)sb.st_size);
		  close(log_fd);

		  n=bytes/sizeof(UDM_LOGWORD);

printf("sorting... "); fflush(stdout);
                  UdmSort(log_buf,n,sizeof(UDM_LOGWORD),(qsort_cmp)cmplog);
		  n=RemoveOldWords(log_buf,n,del_buf,del_count);
		} else {
		  n = 0;
		}

printf("writing... "); fflush(stdout);

		for(i=1;i<n+1;i++){
			/* Next word? */
			if( (i==n) || (log_buf[prev].wrd_id != log_buf[i].wrd_id)){
				if(mkeys==nkeys){
					mkeys+=500;
					keys=(T_WRD_POS_LEN*)realloc(keys,mkeys*sizeof(T_WRD_POS_LEN));
				}
				keys[nkeys].wrd_id=log_buf[prev].wrd_id;
				keys[nkeys].pos=prev;
				keys[nkeys].len=i-prev;
				keys[nkeys].done=0;
				nkeys++;
			
				prev=i;
			}
		}
		for(k=0;k<old_ind_count;k++){
			uint4		wrd_id=old_ind[k].wrd_id;
			UDM_URL_CRD	*new_data=NULL;
			UDM_URL_CRD	*old_data=NULL;
			size_t		old_count;
			size_t		old_added=0; 
			size_t		new_count=0;
			size_t		new_len;
			int		wrd_ind;
			size_t 		o;
			int 		start=0;

			old_data=(UDM_URL_CRD*)malloc(old_ind[k].len);
			if(old_ind[k].pos!=lseek(old_dat_fd,old_ind[k].pos,SEEK_SET)){
				UdmLog_noagent(Env, UDM_LOG_ERROR, "Can't seek '%s': %s\n", old_dat_name, strerror(errno));
				goto err1;
			}
			read(old_dat_fd,old_data,old_ind[k].len);
			new_data=(UDM_URL_CRD*)malloc(old_ind[k].len);

			old_count=old_ind[k].len/sizeof(UDM_URL_CRD);

			for(o=0;o<old_count;o++){
				if(!PresentInDelLog(del_buf,del_count,&start,old_data[o].url_id)){
					new_data[old_added]=old_data[o];
					old_added++;
				}
			}
//printf("del_count=%d\told_count=%d\told_added=%d\n",del_count,old_count,old_added);

			if( (wrd_ind=PresentInNew(keys, nkeys, wrd_id))>=0 ){
				new_count=old_added+keys[wrd_ind].len;
				new_len=new_count*sizeof(UDM_URL_CRD);
				new_data=(UDM_URL_CRD*)realloc(new_data,new_len);
//printf("wrd_ind=%d old_added=%d keys[wrd_ind].pos=%d keys[wrd_ind].len=%d\n",wrd_ind,old_added,keys[wrd_ind].pos,keys[wrd_ind].len);
				if(new_data){
					size_t j;
					for(j=old_added;j<new_count;j++){
						new_data[j].url_id=log_buf[keys[wrd_ind].pos+j-old_added].url_id;
						new_data[j].coord=log_buf[keys[wrd_ind].pos+j-old_added].coord;
					}
				}else{
					goto err1;
				}
				keys[wrd_ind].done=1;
			}else{
				new_count=old_added;
			}
			if(new_count){
				UdmSort(new_data,new_count,sizeof(UDM_URL_CRD),(qsort_cmp)cmp_url_id);
				new_len=new_count*sizeof(UDM_URL_CRD);
				write(new_dat_fd,new_data,new_len);

				if(new_ind_count==new_ind_mcount){
					new_ind_mcount+=500;
					new_ind=realloc(new_ind,new_ind_mcount*sizeof(T_DAT_IND));
				}
				new_ind[new_ind_count].wrd_id=wrd_id;
				new_ind[new_ind_count].pos=pos;
				new_ind[new_ind_count].len=new_len;
				pos+=new_len;
				new_ind_count++;
			}
			UDM_FREE(old_data);
			UDM_FREE(new_data);
		}
		if(old_ind)free(old_ind);
		if(old_dat_fd)close(old_dat_fd);
		for(i=0;i<nkeys;i++){
			if(!keys[i].done){
				UDM_URL_CRD       *new_data=NULL;
				size_t		j;
				
				new_data=(UDM_URL_CRD*)malloc(keys[i].len*sizeof(*new_data));
				for(j=0;j<keys[i].len;j++){
					new_data[j].url_id=log_buf[keys[i].pos+j].url_id;
					new_data[j].coord=log_buf[keys[i].pos+j].coord;
				}
				UdmSort(new_data,keys[i].len,sizeof(UDM_URL_CRD),(qsort_cmp)cmp_url_id);
				write(new_dat_fd,new_data,keys[i].len*sizeof(UDM_URL_CRD));
				if(new_ind_count==new_ind_mcount){
					new_ind_mcount+=500;
					new_ind=realloc(new_ind,new_ind_mcount*sizeof(T_DAT_IND));
				}
				new_ind[new_ind_count].wrd_id=keys[i].wrd_id;
				new_ind[new_ind_count].pos=pos;
				new_ind[new_ind_count].len=keys[i].len*sizeof(UDM_URL_CRD);
				pos+=keys[i].len*sizeof(UDM_URL_CRD);
				new_ind_count++;

				free(new_data);
			}
		}
		free(keys);
	}
	UDM_FREE(log_buf);
	
	close(new_dat_fd);

printf("creating index... "); fflush(stdout);

	UdmSort(new_ind,new_ind_count,sizeof(T_DAT_IND),(qsort_cmp)cmp_ind);	
	write(new_ind_fd,new_ind,new_ind_count*sizeof(T_DAT_IND));
	UDM_FREE(new_ind);
	close(new_ind_fd);

#if (WIN32|WINNT)
	remove(old_dat_name);
	remove(old_ind_name);
#endif
	
	rename(new_dat_name,old_dat_name);
	rename(new_ind_name,old_ind_name);

printf("Done\n"); fflush(stdout);
	
	return(0);

err1:
	if(log_fd)close(log_fd);
	if(old_ind_fd)close(old_ind_fd);
	if(new_ind_fd)close(new_ind_fd);
	if(old_dat_fd)close(old_dat_fd);
	if(new_dat_fd)close(new_dat_fd);

	return(-1);
}

/****************** Search stuff ************************************/
/*
static int cmp_hex8_ind(const UDM_UINT8_POS_LEN *c1, const UDM_UINT8_POS_LEN *c2){
	uint4 n1=c1->hi;
	uint4 n2=c2->hi;

	if(n1==n2){
		n1=c1->lo; 
		n2=c2->lo;
	}
	if(n1<n2) return(-1);
	if(n1>n2) return(1);
	return(0);
}
*/

static int cmp_hex4_ind(const UDM_UINT4_POS_LEN *c1, const UDM_UINT4_POS_LEN *c2){
	if(c1->val<c2->val) return(-1);
	if(c1->val>c2->val) return(1);
	return(0);
}

static void UdmDecodeTimeStr(const char *str, uint4 *from, uint4 *to){
	*from=0;
	*to=0;
}

int UdmAddSearchLimit(UDM_AGENT *Agent, int type, const char *file_name, const char *val){
	uint4 hi,lo;
	
	if(Agent->nlimits == MAX_SEARCH_LIMIT - 1) return(1);
	
	Agent->limits[Agent->nlimits].type = type;
	strcpy(Agent->limits[Agent->nlimits].file_name, file_name);
	switch(type){
		case 0: UdmDecodeHex8Str(val,&hi,&lo); break;
		case 1: UdmDecodeTimeStr(val,&hi,&lo); break;
		case 2: hi=atoi(val); lo=0; break;
		case 3: hi=UdmStrCRC32(val); lo=0; break;
	}	
	Agent->limits[Agent->nlimits].hi = hi;
	Agent->limits[Agent->nlimits].lo = lo;
	
	Agent->nlimits++;
	
	return(0);
}

static int* LoadNestedLimit(UDM_AGENT *Agent,const char *name,uint4 hi,uint4 lo,size_t *size){
	char	fname[1024];
	int	ind_fd,dat_fd;
	UDM_UINT8_POS_LEN *ind=NULL;
	struct	stat sb;
	size_t	num;
	uint4	*data;
	uint4	start,stop,len;
	uint8	next=0;
	
	if(hi==0&&lo==0)return(NULL);

	snprintf(fname, sizeof(fname), "%s%s%c%s.ind", Agent->Conf->vardir, TREEDIR, UDMSLASH, name);
	if((ind_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	fstat(ind_fd, &sb);
	ind=(UDM_UINT8_POS_LEN*)malloc((size_t)sb.st_size);
	if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(ind_fd);
	num=sb.st_size/sizeof(UDM_UINT8_POS_LEN);		

	{
		uint4 l=0,r=num,m;

		while(l<r){
			m=(l+r)/2;
			if(ind[m].hi<hi) l=m+1;
			else
			if((ind[m].hi==hi)&&(ind[m].lo<lo)) l=m+1;
			else r=m;
		}
		if(r==num) goto err1;
		else start=r;
	}	

	if(ind[start].hi>=0xFF000000){
		next=0xFFFFFFFFFFFFFFFF;
	}else{
		uint8 a=((uint8)(ind[start].hi)<<32) + (ind[start].lo);
		uint8 b;
		int i;
		
		for(i=7*8;i>=0;i-=8){
			if((b=a>>i)){
				next=(b+1)<<i;
				break;
			}
		}
	}

	stop=start;
	while(1){
		stop++;
		if(stop==(num-1)) break;
		else
		if((stop>(num-1)) ||
		   (ind[stop].hi>(next>>32)) ||
		   (ind[stop].hi==(next>>32) && ind[stop].lo>(next&&0xFFFFFFFF))){
			stop--;
			break;
		}
		 
	}
	snprintf(fname, sizeof(fname), "%s%s%c%s.dat", Agent->Conf->vardir, TREEDIR, UDMSLASH, name);
	if((dat_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	if(ind[start].pos!=lseek(dat_fd,ind[start].pos,SEEK_SET)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't seek '%s': %s", fname, strerror(errno));
		goto err1;
	}
	len=ind[stop].pos+ind[stop].len-ind[start].pos;
	data=(uint4*)malloc(len);
	if(len!=read(dat_fd,data,len)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(dat_fd);
	
	free(ind);

	UdmSort(data,len/4,sizeof(uint4),(qsort_cmp)cmp_uint4);
	*size=len/sizeof(uint4);
	return((int*)data);

err1:
	if(ind)free(ind);
	return(NULL);
}

static int* LoadLinearLimit(UDM_AGENT *Agent,const char *name,uint4 val,size_t *size){
	char	fname[1024];
	int	ind_fd,dat_fd;
	UDM_UINT4_POS_LEN key,*found,*ind=NULL;
	struct	stat sb;
	size_t	num;
	uint4	*data;

	snprintf(fname, sizeof(fname), "%s%s%c%s.ind", Agent->Conf->vardir, TREEDIR, UDMSLASH, name);
	if((ind_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	fstat(ind_fd, &sb);
	ind=(UDM_UINT4_POS_LEN*)malloc((size_t)sb.st_size);
	if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(ind_fd);

	num=sb.st_size/sizeof(UDM_UINT4_POS_LEN);
	key.val=val;
	if(!(found=bsearch(&key,ind,num,sizeof(UDM_UINT4_POS_LEN),(qsort_cmp)cmp_hex4_ind))){
		goto err1;
	}
	snprintf(fname, sizeof(fname), "%s%s%c%s.dat", Agent->Conf->vardir, TREEDIR, UDMSLASH, name);
	if((dat_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	if(found->pos!=lseek(dat_fd,found->pos,SEEK_SET)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't seek '%s': %s", fname, strerror(errno));
		goto err1;
	}
	data=(uint4*)malloc(found->len);
	if(found->len!=read(dat_fd,data,found->len)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(dat_fd);

	free(ind);

	*size=found->len/sizeof(uint4);
	return((int*)data);

err1:
	if(ind)free(ind);
	return(NULL);
}

static int* LoadTimeLimit(UDM_AGENT *Agent,const char *name,uint4 from,uint4 to,size_t *size){
	char	fname[1024];
	int	ind_fd,dat_fd;
	UDM_UINT4_POS_LEN *found1,*found2,*ind=NULL;
	struct	stat sb;
	size_t	num,len;
	uint4	*data;
	
	if((from)&&(to)&&(from>to)) return(NULL);

	snprintf(fname, sizeof(fname), "%s%s%c%s.ind", Agent->Conf->vardir, TREEDIR, UDMSLASH, name);
	if((ind_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	fstat(ind_fd, &sb);
	ind=(UDM_UINT4_POS_LEN*)malloc((size_t)sb.st_size);
	if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(ind_fd);

	num=sb.st_size/sizeof(UDM_UINT4_POS_LEN);

	if(!from){
		found1=&ind[0];
	}else{
		uint4 l=0,r=num,m;

		while(l<r){
			m=(l+r)/2;
			if(ind[m].val<from) l=m+1;
			else r=m;
		}
		if(r==num) found1=NULL;
		found1=&ind[r];
	}	
	if(!to){
		found2=&ind[num-1];
	}else{
		uint4 l=0,r=num,m;

		while(l<r){
			m=(l+r)/2;
			if(ind[m].val<to) l=m+1;
			else r=m;
		}
		if(r==num) found2=&ind[num-1];
		else if(ind[r].val==to) found2=&ind[r];
		else if(l>0) found2=&ind[l-1];
		else found2=NULL;
	}	
	if(!found1||!found2)return(NULL);

	snprintf(fname, sizeof(fname), "%s%s%c%s.dat", Agent->Conf->vardir, TREEDIR, UDMSLASH, name);
	if((dat_fd=open(fname,O_RDONLY|UDM_BINARY))<0){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't open '%s': %s", fname, strerror(errno));
		goto err1;
	}
	if(found1->pos!=lseek(dat_fd,found1->pos,SEEK_SET)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't seek '%s': %s", fname, strerror(errno));
		goto err1;
	}
	len=found2->pos+found2->len-found1->pos;
	data=(uint4*)malloc(len);
	if(len!=read(dat_fd,data,len)){
		UdmLog(Agent, UDM_LOG_ERROR, "Can't read '%s': %s", fname, strerror(errno));
		goto err1;
	}
	close(dat_fd);
	
	free(ind);
	
	UdmSort(data,len/4,sizeof(uint4),(qsort_cmp)cmp_uint4);
	*size=len/sizeof(uint4);
	return((int*)data);

err1:
	if(ind)free(ind);
	return(NULL);
}

static int PresentInLimit(uint4 *buf,size_t buf_count,size_t *start,const int url_id){
	register size_t m;
	register size_t l;
	register size_t r;

	if(start)l=*start;
	else l=0;
	r=buf_count;
	while(l<r){
		m=(l+r)/2;
		if(buf[m]<url_id) l=m+1;
		else r=m;
	}
	if(start)*start=r;
	if(r==buf_count) return(0);
	else
		if(buf[r]==url_id) return(1);
		else return(0);
}

typedef struct {
	int last;
	int cur;
	int num;
} UDM_MERG;
                 
typedef struct {
	UDM_URL_CRD *plast;
	UDM_URL_CRD *pcur;
	int num;
	int count;
} UDM_PMERG;

#define MAXMERGE 256 /* <= one byte */

int UdmFindCache(UDM_AGENT * Indexer,UDM_RESULT *Res){
	int i;
	int nmerg=0;
	UDM_URLCRDLIST MCoordList;
	UDM_MERG merg[MAXMERGE];
	UDM_PMERG pmerg[MAXMERGE];
	int ind_fd=0,dat_fd=0;
	int wf[256];
	
	uint4 *lim_data[MAX_SEARCH_LIMIT];
	uint4 lim_size[MAX_SEARCH_LIMIT];
	uint4 lim_start[MAX_SEARCH_LIMIT];
	uint4 nlims=0;
	
	int page_number;
	int page_size;
	int search_mode = UdmSearchMode(UdmVarListFindStr(&Indexer->Conf->Vars, "m", "all"));

#ifdef DEBUG_SEARCH
	unsigned long ticks;
	unsigned long total_ticks=UdmStartTimer();
	
	UdmLog(Indexer, UDM_LOG_DEBUG, "Start UdmFindCache()");
#endif
	
	UdmWeightFactorsInit(UdmVarListFindStr(&Indexer->Conf->Vars,"wf",""),wf);
	bzero(&MCoordList,sizeof(MCoordList));
	bzero(lim_start,MAX_SEARCH_LIMIT*sizeof(uint4));
	
#ifdef DEBUG_SEARCH
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Reading limits (%d)... ", Indexer->nlimits);
	ticks=UdmStartTimer();
#endif
	for(i = 0; i < Indexer->nlimits; i++){
		switch(Indexer->limits[i].type){
			case 0:	if((lim_data[nlims] = LoadNestedLimit(Indexer,Indexer->limits[i].file_name,
								Indexer->limits[i].hi,
								Indexer->limits[i].lo,
								&lim_size[nlims]))) nlims++;
				else return(-1);
				break;
			case 1: if((lim_data[nlims] = LoadTimeLimit(Indexer,Indexer->limits[i].file_name,
								Indexer->limits[i].hi,
								Indexer->limits[i].lo,
								&lim_size[nlims]))) nlims++;
				else return(-1);
				break;
			case 2:
			case 3: if((lim_data[nlims] = LoadLinearLimit(Indexer,Indexer->limits[i].file_name,
								Indexer->limits[i].hi,
								&lim_size[nlims]))) nlims++;
				else return(-1);
				break;
		}
	}
#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\t\t\tDone (%.2f)", (float)ticks / 1000);
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Reading .wrd files (%d words)... ", Res->WWList.nwords);
	ticks=UdmStartTimer();
#endif
	for(i=0;i<Res->WWList.nwords;i++){
		size_t num;
		char ind_name[1024];
		char dat_name[1024];
		T_DAT_IND *ind=NULL;
		T_DAT_IND key, *found;
		
		struct stat sb;

		if (Res->WWList.Word[i].origin == UDM_WORD_ORIGIN_STOP) continue;

		snprintf(ind_name, sizeof(ind_name), "%s%s%c%03X%s", Indexer->Conf->vardir, TREEDIR, UDMSLASH,
			 ((uint4)Res->WWList.Word[i].crcword)>>20, ".ind");
		if((ind_fd=open(ind_name,O_RDONLY|UDM_BINARY))<0){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't open '%s': %s", ind_name, strerror(errno));
			continue;
		}
		fstat(ind_fd, &sb);
		ind=(T_DAT_IND*)malloc((size_t)sb.st_size);
		if(sb.st_size!=read(ind_fd,ind,(size_t)sb.st_size)){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't read '%s': %s", ind_name, strerror(errno));
			close(ind_fd); ind_fd=0;
			goto cont;
		}
		close(ind_fd); ind_fd=0;

		num=sb.st_size/sizeof(T_DAT_IND);		
		key.wrd_id=Res->WWList.Word[i].crcword;
		if(!(found=bsearch(&key,ind,num,sizeof(T_DAT_IND),(qsort_cmp)cmp_ind))){
			goto cont;
		}
		snprintf(dat_name, sizeof(dat_name), "%s%s%c%03X%s", Indexer->Conf->vardir, TREEDIR, UDMSLASH,
			 ((uint4)Res->WWList.Word[i].crcword)>>20, ".dat");
		if((dat_fd=open(dat_name,O_RDONLY|UDM_BINARY))<0){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't open '%s': %s", dat_name, strerror(errno));
			close(dat_fd); dat_fd=0;
			goto cont;
		}
		if(found->pos!=lseek(dat_fd,found->pos,SEEK_SET)){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't seek '%s': %s", dat_name, strerror(errno));
			close(dat_fd); dat_fd=0;
			goto cont;
		}
		Res->CoordList.Coords=(UDM_URL_CRD*)realloc(Res->CoordList.Coords,
			Res->CoordList.ncoords*sizeof(UDM_URL_CRD)+found->len);
		
		if(found->len!=read(dat_fd,Res->CoordList.Coords+Res->CoordList.ncoords,found->len)){
			UdmLog(Indexer, UDM_LOG_EXTRA, "Can't read '%s': %s", dat_name, strerror(errno));
			close(dat_fd); dat_fd=0;
			goto cont;
		}
		close(dat_fd); dat_fd=0;
		
		num=found->len/sizeof(UDM_URL_CRD);
		Res->WWList.Word[i].count = num;

		if(num){
			merg[nmerg].cur=Res->CoordList.ncoords;
			merg[nmerg].last=Res->CoordList.ncoords+num-1;
			merg[nmerg].num=i;
			nmerg++;
		}

		Res->CoordList.ncoords+=num;
cont:
		free(ind);
	}
	
#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\tDone (%.2f)", (float)ticks / 1000);
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Merging (%d groups, %d urls)... ", nmerg, Res->CoordList.ncoords);
	ticks=UdmStartTimer();
#endif

	if (search_mode == UDM_MODE_ALL) {

#ifdef MODE_ALL_VIA_LIMITS

	  int n_total = 0, min_total = 0, icurrent = 0, corder = 0, nid, z;
	  UDM_URL_CRD *p;

	  for (i = 0; (i < Res->WWList.nwords) && (Res->WWList.Word[i].origin == UDM_WORD_ORIGIN_STOP); i++);
	  corder = Res->WWList.Word[i].order;
	  for( ; (i < Res->WWList.nwords) && (Res->WWList.Word[i].order == corder); i++) {
	    min_total += Res->WWList.Word[i].count;
	  }
	  for(i = 0; (i < Res->WWList.nwords); i++) {
	    if (Res->WWList.Word[i].origin == UDM_WORD_ORIGIN_STOP) continue;
	    if(Res->WWList.Word[i].order == corder) {
	      n_total += Res->WWList.Word[i].count;
	    } else {
	      if (min_total > n_total) {
		min_total = n_total;
	      }
	      corder = Res->WWList.Word[i].order;
	      n_total = Res->WWList.Word[i].count;
	    }
	  }
	  if (min_total > n_total) {
	    min_total = n_total;
	  }
	  if (min_total == 0) {
	    Res->CoordList.ncoords = 0;
	    UDM_FREE(Res->CoordList.Coords);
	    Res->total_found = 0;
	    return 0;
	  }

	  corder = Res->WWList.Word[merg[0].num].order;
	  n_total = 0;
	  p = Res->CoordList.Coords + merg[0].cur;
	  for(i = 0; (i < nmerg); i++) {
	    if (Res->WWList.Word[merg[i].num].order == corder) {
	      n_total += Res->WWList.Word[merg[i].num].count;
	    } else {
		  p = Res->CoordList.Coords + merg[icurrent].cur;
		  lim_data[nlims] = (uint4*)malloc( n_total * sizeof(uint4));
		  nid = 0;
		  for (z = 0; z < n_total; z++) {
		    lim_data[nlims][nid] = p->url_id;
		    if ((nid == 0) || (lim_data[nlims][nid] != lim_data[nlims][nid-1])) {
		      nid++;
		    }
		    p++;
		  }
		  UdmSort(lim_data[nlims], (size_t)nid, sizeof(uint4), (qsort_cmp)cmp_uint4);
		  lim_size[nlims] = nid;
		  lim_start[nlims] = 0;
		  nlims++;
		
	      icurrent = i;
	      corder = Res->WWList.Word[merg[i].num].order;
	      n_total = Res->WWList.Word[merg[i].num].count;
	    }
	  }
	    p = Res->CoordList.Coords + merg[icurrent].cur;

	    lim_data[nlims] = (uint4*)malloc(n_total * sizeof(uint4));
	    nid = 0;
	    for (z = 0; z < n_total; z++) {
	      lim_data[nlims][nid] = p->url_id;
	      if ((nid == 0) || (lim_data[nlims][nid] != lim_data[nlims][nid-1])) {
		nid++;
	      }
	      p++;
	    }
	    UdmSort(lim_data[nlims], (size_t)nid, sizeof(uint4), (qsort_cmp)cmp_uint4);
	    lim_size[nlims] = nid;
	    lim_start[nlims] = 0;
	    nlims++;

#else /* MODE_ALL_VIA_MODE_BOOL */
	    int z;
	    UdmVarListReplaceStr(&Indexer->Conf->Vars, "m", "bool"); 
	    for(z = 0; z < Res->WWList.nuniq; z++) {
	      if (z) {
		Res->items[Res->nitems].cmd = UDM_STACK_AND;
		Res->items[Res->nitems].arg = 0;
		Res->nitems++;
	      }
	      Res->items[Res->nitems].cmd=UDM_STACK_WORD;
	      Res->items[Res->nitems].arg = 1L << (z);
	      Res->nitems++;
	    }

#endif

	}

	if(Res->CoordList.ncoords){
		/* Allocate memory for merged words */
		MCoordList.Coords=(UDM_URL_CRD*)malloc(Res->CoordList.ncoords*sizeof(UDM_URL_CRD));

		for(i=0;i<nmerg;i++){
			pmerg[i].pcur  = Res->CoordList.Coords+merg[i].cur;
			pmerg[i].plast = Res->CoordList.Coords+merg[i].last;
			pmerg[i].num   = Res->WWList.Word[merg[i].num].order;
			pmerg[i].count = 0;
		}
	}
	while(nmerg && Res->CoordList.ncoords){
		UDM_PMERG *p;

		if(nmerg==1){
			/*
			if(nlims==0){
				size_t num;

				num=pmerg[0].plast-pmerg[0].pcur;
				memcpy(&wrd1[nwrd1],pmerg[0].pcur,num*sizeof(*wrd1));
				nwrd1+=num;
			}else
			*/
			for(p=&pmerg[0];(p->pcur)<=(p->plast);(p->pcur)++){
				int l;
				uint4 weight;
				
				weight=wf[UDM_WRDSEC(p->pcur->coord)];

				if(weight){
					for(l=0;l<nlims;l++){
						if(!PresentInLimit(lim_data[l],lim_size[l],&lim_start[l],p->pcur->url_id)){
							weight=0;
							break;
						}
					}
				}
				if(weight){
					UDM_URL_CRD *Crd = MCoordList.Coords + MCoordList.ncoords;
					*Crd=*(p->pcur);
					Crd[0].coord &= 0xFFFF0000;
					Crd[0].coord += ((weight<<8)+p->num);
					MCoordList.ncoords++;
					p->count++;
				}
			}
			break;
		}else{
			size_t k;
			uint4 min_url_id=(pmerg[0].pcur)->url_id;
			uint4 min_pos=UDM_WRDPOS((pmerg[0].pcur)->coord);
			uint4 min_num=0;
			int l;
			uint4 weight;

			for(k=1;k<nmerg;k++){ 
				/* note ">=" to compare pos in the same url */
				if( (min_url_id>(pmerg[k].pcur)->url_id) ||
				    ((min_url_id==(pmerg[k].pcur)->url_id)&&
				     (min_pos>UDM_WRDPOS((pmerg[k].pcur)->coord))) ){
					min_url_id=(pmerg[k].pcur)->url_id;
					min_pos=UDM_WRDPOS((pmerg[k].pcur)->coord);
					min_num=k;
				}
			}
			p=&pmerg[min_num];
			weight=wf[UDM_WRDSEC(p->pcur->coord)];
			if(weight){
				for(l=0;l<nlims;l++){
					if(!PresentInLimit(lim_data[l],lim_size[l],&lim_start[l],p->pcur->url_id)){
						weight=0;
						break;
					}
				}
			}
			if(weight){
				UDM_URL_CRD *Crd = MCoordList.Coords + MCoordList.ncoords;
				Crd[0]=*(p->pcur);
				Crd[0].coord &= 0xFFFF0000;
				Crd[0].coord += ((weight<<8)+p->num);
				MCoordList.ncoords++;
				p->count++;
			}
			
			if(p->pcur==p->plast){
				nmerg--;
				if(min_num<nmerg){
					memmove(p,p+1,sizeof(UDM_PMERG)*(nmerg-min_num));
				}
			}else{
				(p->pcur)++;
			}
		}
	}
	UDM_FREE(Res->CoordList.Coords);
	Res->CoordList = MCoordList;
	
	for(i=0;i<nlims;i++) UDM_FREE(lim_data[i]);

#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\tDone (%.2f) Merged ncoords1=%d", (float)ticks / 1000, MCoordList.ncoords);
	UdmLog(Indexer, UDM_LOG_DEBUG, "    Grouping by url_id... ");
	ticks=UdmStartTimer();
#endif

	UdmGroupByURL(Indexer,Res);
	
#ifdef DEBUG_SEARCH
	ticks=UdmStartTimer() - ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "\t\tDone (%.2f)", (float)ticks / 1000);
	total_ticks=UdmStartTimer() - total_ticks;
	UdmLog(Indexer, UDM_LOG_DEBUG, "Stop UdmFindCache() - %.2f\n", (float)total_ticks / 1000);
#endif

	page_size   = UdmVarListFindInt(&Indexer->Conf->Vars, "ps", 20);
	page_number = UdmVarListFindInt(&Indexer->Conf->Vars, "np", 0);

	Res->total_found = Res->CoordList.ncoords;
	Res->first = page_number * page_size;	
	if (Res->first >= Res->total_found) Res->first = (Res->total_found) ? Res->total_found - 1 : 0;

	/* If results more than 1 page */
	/* we must cut the tail        */
	if((Res->first + page_size) > Res->total_found){
		Res->num_rows = Res->total_found-Res->first;
	}else{
		Res->num_rows = page_size;
	}
	Res->last = Res->first + Res->num_rows - 1;

	/* first and last begins from 0, make it begin from 1 */
	Res->first++;
	Res->last++;

	/* Allocate an array for documents information */
	Res->Doc=(UDM_DOCUMENT*)UdmXmalloc(sizeof(UDM_DOCUMENT)*(Res->num_rows));

	/* Copy url_id and coord to result */
	for(i = 0; i < Res->num_rows; i++){
		UdmDocInit(&Res->Doc[i]);
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"ID",Res->CoordList.Coords[i + Res->first * Res->offset].url_id);
		UdmVarListReplaceInt(&Res->Doc[i].Sections,"Score",(int)Res->CoordList.Coords[i + Res->first * Res->offset].coord);
	}

	return(0);
	
err1:
	if(ind_fd)close(ind_fd);
	if(dat_fd)close(dat_fd);
	return(-1);
}

/************************** Logd stuff **************************/

static int LogdSaveBuf(UDM_DB *db,int num){
	int		fd;
	size_t		nbytes;
	UDM_LOGD	*logd=&db->LOGD;
	char		fname[1024];
	
	snprintf(fname, sizeof(fname), "%s%03X.log", db->log_dir, num);
	
	if((fd=open(fname,open_flags,open_perm))!=-1){
		nbytes=logd->wrd_buf[num].nrec*sizeof(UDM_LOGWORD);
		if(nbytes!=write(fd,logd->wrd_buf[num].data,nbytes)){
			sprintf(db->errstr,"Can't write '%s': %s\n",fname,strerror(errno));
			db->errcode=1;
			return UDM_ERROR;
		}
		close(fd);
		logd->wrd_buf[num].nrec=0;
	}else{
		sprintf(db->errstr,"Can't open '%s': %s\n",fname,strerror(errno));
		db->errcode=1;
		return UDM_ERROR;
	}
	return UDM_OK;
}

int LogdSaveAllBufs(UDM_DB *db){
	int i;
	
	for(i=0;i<MAX_LOG;i++){
		if(db->LOGD.wrd_buf[i].nrec){
			if(UDM_OK!=LogdSaveBuf(db,i)) return UDM_ERROR;
		}
	}
	return UDM_OK;
}

static int LogdOpenLogs(UDM_DB *db){
	char	del_log_name[1024];
	int	i;
	
	for(i=0;i<MAX_LOG;i++)db->LOGD.wrd_buf[i].nrec=0;
	
	snprintf(del_log_name,sizeof(del_log_name),"%s%s", db->log_dir, "del.log");
	if((db->del_fd=open(del_log_name,O_RDWR | O_CREAT | UDM_BINARY, 0600))==-1){
		sprintf(db->errstr,"Can't open '%s': %s\n",del_log_name,strerror(errno));
		db->errcode=1;
		return UDM_ERROR;
	}
	return UDM_OK;
}

static int LogdCloseLogs(UDM_DB *db){
	if(db->del_fd){
		close(db->del_fd);
		db->del_fd=0;
	}
	return LogdSaveAllBufs(db);
}

int LogdInit(UDM_DB *db,const char* var_dir){
	sprintf(db->log_dir,"%s%s%s",var_dir,SPLDIR,UDMSLASHSTR);
	db->errstr[0]=0;
	return LogdOpenLogs(db);
}

int LogdClose(UDM_DB *db){
	return LogdCloseLogs(db);
}

int LogdStoreDoc(UDM_DB *db,UDM_LOGD_CMD cmd,UDM_LOGD_WRD *wrd){
	UDM_LOGDEL logdel;
	UDM_LOGD *logd=&db->LOGD;
	int i;
	
	logdel.stamp=cmd.stamp;
	logdel.url_id=cmd.url_id;
	if(sizeof(UDM_LOGDEL)!=write(db->del_fd,&logdel,sizeof(UDM_LOGDEL))){
		sprintf(db->errstr,"Can't write to del.log: %s\n",strerror(errno));
		db->errcode=1;
		return UDM_ERROR;
	}
	
	if(cmd.nwords==0)return UDM_OK;

	for(i=0;i<cmd.nwords;i++){
		int	num=wrd[i].wrd_id>>20;
		int	nrec=logd->wrd_buf[num].nrec;
		
		logd->wrd_buf[num].data[nrec].stamp=cmd.stamp;
		logd->wrd_buf[num].data[nrec].url_id=cmd.url_id;
		logd->wrd_buf[num].data[nrec].wrd_id=wrd[i].wrd_id;
		logd->wrd_buf[num].data[nrec].coord=wrd[i].coord;
		logd->wrd_buf[num].nrec++;
		if(logd->wrd_buf[num].nrec==MAX_WRD){
			if(UDM_OK!=LogdSaveBuf(db,num)) return UDM_ERROR;
		}
	}
	return UDM_OK;
}

/************************** /Logd stuff **************************/
